#!/bin/bash

# Define the dialog exit status codes
: ${DIALOG_OK=0}
: ${DIALOG_CANCEL=1}
: ${DIALOG_HELP=2}
: ${DIALOG_EXTRA=3}
: ${DIALOG_ITEM_HELP=4}
: ${DIALOG_ESC=255}

show_menu() {
    menu_choice=$(dialog --clear --title "XPU-Prof Signal Configuration" \
    --ok-label "Select" \
    --cancel-label "Exit" \
    --menu "Choose Event:" 15 50 10 \
    1 "Add Driver Event" \
    2 "Add SSE Event" \
    3 "Add Cluster Event" \
    4 "Add SDNN Event" \
    5 "Add SDNN Cluster Event" \
    6 "Add Cache/HBM Event" \
    7 "Add SDMA Event" \
    8 "Add PCIe/C2C Event" \
    9 "Add SMMU Event" \
    10 "Exit" 2>&1 1>&3)

    return_value=$?
}

show_add_result() {
    dialog --title "Message Box" --msgbox  "Add signals to xpu-prof.gen-settings.json!." 10 60
}

show_add_error() {
    dialog --title "Message Box" --msgbox  "Invalid Configuration!." 10 60
}

##################################driver event##################################

insert_driver_event() {
    if [ -n "$driver_events" ]; then
        for event in $driver_events; do
            json=$(echo "\"name\": \"$event\"")
            echo "      {" >> "$file"
            echo "          $json" >> "$file"
            echo "      }," >> "$file"
        done
        show_add_result
    else
        show_add_error
    fi
}

add_driver_event() {
    driver_events=$(dialog --clear --title "Driver Event Configuration" \
    --ok-label "Add" \
    --cancel-label "Back" \
    --checklist "Select driver signals to track:" 15 50 6 \
    "DriverWait" "" off \
    "DriverLaunch" "" off \
    "DriverMemcpy" "" off \
    "DriverEventWait" "" off \
    "DriverEventRecord" "" off \
    "DriverEventStreamWait" "" off 2>&1 1>&3)

    return_value=$?
    if [ $return_value != $DIALOG_OK ]; then
        return
    fi
    insert_driver_event
}

choose_device() {
    devices=$(dialog --clear --title "XPU-Prof Event Configuration" \
    --ok-label "Next" \
    --cancel-label "Cancel" \
    --checklist "Choose Device Id:" 15 50 8 \
    0 "device 0" off \
    1 "device 1" off \
    2 "device 2" off \
    3 "device 3" off \
    4 "device 4" off \
    5 "device 5" off \
    6 "device 6" off \
    7 "device 7" off 2>&1 1>&3)

    return_value=$?
}

choose_channel() {
    channels=$(dialog --clear --title "SSE Event Configuration" \
    --colors \
    --ok-label "Next" \
    --cancel-label "Cancel" \
    --checklist "Choose Channel Id (\Z1ALL channels enabled if select NONE\Zn):" 15 50 16 \
    0 "channel 0" off \
    1 "channel 1" off \
    2 "channel 2" off \
    3 "channel 3" off \
    4 "channel 4" off \
    5 "channel 5" off \
    6 "channel 6" off \
    7 "channel 7" off \
    8 "channel 8" off \
    9 "channel 9" off \
    10 "channel 10" off \
    11 "channel 11" off \
    12 "channel 12" off \
    13 "channel 13" off \
    14 "channel 14" off \
    15 "channel 15" off 2>&1 1>&3)

    return_value=$?
}

choose_arch() {
    arch=$(dialog --clear --title "XPU-Prof Event Configuration" \
    --ok-label "Next" \
    --cancel-label "Cancel" \
    --menu "Choose Device Arch:" 15 50 2 \
    3 "Kunlun 3" \
    4 "Mars" 2>&1 1>&3)

    return_value=$?
}

##################################sse event##################################

add_kl3_sse() {
    sse_events=$(dialog --clear --title "KL3 SSE Event Configuration" \
    --ok-label "Add" \
    --cancel-label "Back" \
    --checklist "Select sse signals to track:" 15 50 3 \
    "SSEduration" "" off \
    "SSEinterrupt" "" off \
    "SSEevent" "" off 2>&1 1>&3)

    return_value=$?
}

add_kl4_sse() {
    sse_events=$(dialog --clear --title "Mars SSE Event Configuration" \
    --ok-label "Add" \
    --cancel-label "Back" \
    --checklist "Select sse signals to track:" 15 50 3 \
    "SSEduration" "" off \
    "SSEinterrupt" "" off \
    "SSEenqueue" "" off 2>&1 1>&3)

    return_value=$?
}

insert_sse_event() {
    if [ -n "$sse_events" ]; then
        for device in $devices; do
            for channel in $channels; do
                for event in $sse_events; do
                    bid=$(echo "\"board-id\": \"$device\",")
                    cid=$(echo "\"channel-id\": \"$channel\",")
                    name=$(echo "\"name\": \"$event\"")
                    echo "      {" >> "$file"
                    echo "          $bid" >> "$file"
                    echo "          $cid" >> "$file"
                    echo "          $name" >> "$file"
                    echo "      }," >> "$file"
                done
            done
        done
        show_add_result
    else
        show_add_error
    fi
}

add_sse_event() {
    choose_device
    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    if [ -z "$devices" ]; then
        show_add_error
        return
    fi

    choose_channel
    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    if [ -z "$channels" ]; then
        channels=$(seq -s ' ' 0 15)
    fi

    choose_arch
    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    case $arch in
        3) add_kl3_sse ;;
        4) add_kl4_sse ;;
        *) show_add_error
            return ;;
    esac

    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    insert_sse_event
}

##################################cluster event##################################

choose_kl3_unit() {
    units=$(dialog --clear --title "Cluster Event Configuration" \
    --colors \
    --ok-label "Next" \
    --cancel-label "Cancel" \
    --checklist "Choose Unit Id:" 15 50 12 \
    0 "unit 0" off \
    1 "unit 1" off \
    2 "unit 2" off \
    3 "unit 3" off \
    4 "unit 4" off \
    5 "unit 5" off \
    6 "unit 6" off \
    7 "unit 7" off \
    8 "unit 8" off \
    9 "unit 9" off \
    10 "unit 10" off \
    11 "unit 11" off \
    12 "unit 12" off 2>&1 1>&3)

    return_value=$?
}

choose_kl3_core() {
    cores=$(dialog --clear --title "Cluster Event Configuration" \
    --colors \
    --ok-label "Next" \
    --cancel-label "Cancel" \
    --checklist "Choose Core Id(\Z1ALL cores enabled if select NONE\Zn):" 15 50 4 \
    0 "unit 0" off \
    1 "unit 1" off \
    2 "unit 2" off \
    3 "unit 3" off 2>&1 1>&3)

    return_value=$?
}

insert_kl3_cluster_event() {
    if [ -n "$cluster_events" ]; then
        for device in $devices; do
            for unit in $units; do
                for core in $cores; do
                    for event in $cluster_events; do
                        bid=$(echo "\"board-id\": \"$device\",")
                        uid=$(echo "\"unit-id\": \"$unit\",")
                        cid=$(echo "\"core-id\": \"$core\",")
                        name=$(echo "\"name\": \"$event\"")
                        echo "      {" >> "$file"
                        echo "          $bid" >> "$file"
                        echo "          $uid" >> "$file"
                        echo "          $cid" >> "$file"
                        echo "          $name" >> "$file"
                        echo "      }," >> "$file"
                    done
                done
            done
        done
        show_add_result
    else
        show_add_error
    fi
}

add_kl3_cluster() {
    choose_kl3_unit

    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    if [ -z "$units" ]; then
        show_add_error
        return
    fi

    choose_kl3_core

    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    if [ -z "$cores" ]; then
        cores=$(seq -s ' ' 0 3)
    fi

    cluster_events=$(dialog --clear --title "KL3 Cluster Event Configuration" \
    --ok-label "Add" \
    --cancel-label "Back" \
    --checklist "Select cluster signals to track:" 15 50 9 \
    "ClusterDuration" "" off \
    "ClusterSimdBusy" "" off \
    "ClusterSimdWorking" "" off \
    "ClusterDmaReadBusy" "" off \
    "ClusterDmaReadWorking" "" off \
    "ClusterDmaReadAck" "" off \
    "ClusterDmaWriteBusy" "" off \
    "ClusterDmaWriteWorking" "" off \
    "ClusterPc" "" off 2>&1 1>&3)

    return_value=$?

    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    insert_kl3_cluster_event
}

choose_kl4_unit() {
    units=$(dialog --clear --title "Cluster Event Configuration" \
    --colors \
    --ok-label "Next" \
    --cancel-label "Cancel" \
    --checklist "Choose cluster-level Id(\Z1ALL units enabled if select NONE\Zn):" 15 50 12 \
    0 "cluster 0" off \
    1 "cluster 1" off 2>&1 1>&3)

    return_value=$?
}

choose_kl4_pu() {
    pus=$(dialog --clear --title "Cluster Event Configuration" \
    --colors \
    --ok-label "Next" \
    --cancel-label "Cancel" \
    --checklist "Choose Pu-level Id(\Z1ALL pus enabled if select NONE\Zn):" 15 50 4 \
    0 "pu 0" off \
    1 "pu 1" off \
    2 "pu 2" off \
    3 "pu 3" off 2>&1 1>&3)

    return_value=$?
}

choose_kl4_core() {
    cores=$(dialog --clear --title "Cluster Event Configuration" \
    --colors \
    --ok-label "Next" \
    --cancel-label "Cancel" \
    --checklist "Choose Core-level Id (\Z1ALL cores enabled if select NONE\Zn):" 15 50 16 \
    0 "core 0" off \
    1 "core 1" off \
    2 "core 2" off \
    3 "core 3" off \
    4 "core 4" off \
    5 "core 5" off \
    6 "core 6" off \
    7 "core 7" off \
    8 "core 8" off \
    9 "core 9" off \
    10 "core 10" off \
    11 "core 11" off \
    12 "core 12" off \
    13 "core 13" off \
    14 "core 14" off \
    15 "core 15" off 2>&1 1>&3)

    return_value=$?
}

insert_kl4_cluster_event() {
    flag=0
    if [ -n "$cluster_events" ]; then
        for device in $devices; do
            for unit in $units; do
                for event in $cluster_events; do
                    bid=$(echo "\"board-id\": \"$device\",")
                    uid=$(echo "\"unit-id\": \"$unit\",")
                    name=$(echo "\"name\": \"$event\"")
                    echo "      {" >> "$file"
                    echo "          $bid" >> "$file"
                    echo "          $uid" >> "$file"
                    echo "          $name" >> "$file"
                    echo "      }," >> "$file"
                done
            done
        done
        flag=1
    fi

    if [ -n "$cluster_pus" ]; then
        for device in $devices; do
            for unit in $units; do
                for pu in $pus; do
                    for event in $cluster_pus; do
                        bid=$(echo "\"board-id\": \"$device\",")
                        uid=$(echo "\"unit-id\": \"$unit\",")
                        cid=$(echo "\"core-id\": \"$pu\",")
                        name=$(echo "\"name\": \"$event\"")
                        echo "      {" >> "$file"
                        echo "          $bid" >> "$file"
                        echo "          $uid" >> "$file"
                        echo "          $cid" >> "$file"
                        echo "          $name" >> "$file"
                        echo "      }," >> "$file"
                    done
                done
            done
        done
        flag=1
    fi

    if [ -n "$cluster_cores" ]; then
        for device in $devices; do
            for unit in $units; do
                for core in $cores; do
                    for event in $cluster_cores; do
                        bid=$(echo "\"board-id\": \"$device\",")
                        uid=$(echo "\"unit-id\": \"$unit\",")
                        cid=$(echo "\"core-id\": \"$core\",")
                        name=$(echo "\"name\": \"$event\"")
                        echo "      {" >> "$file"
                        echo "          $bid" >> "$file"
                        echo "          $uid" >> "$file"
                        echo "          $cid" >> "$file"
                        echo "          $name" >> "$file"
                        echo "      }," >> "$file"
                    done
                done
            done
        done
        flag=1
    fi

    if [ $flag == 1 ]; then
        show_add_result
    else
        show_add_error
    fi
}

add_kl4_cluster() {
    choose_kl4_unit

    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    if [ -z "$units" ]; then
        units=$(seq -s ' ' 0 1)
    fi

    cluster_events=$(dialog --clear --title "Mars Cluster Event Configuration" \
    --ok-label "Next" \
    --cancel-label "Back" \
    --checklist "Select cluster-level signals to track:" 15 50 4 \
    "ClusterDuration" "" off \
    "ClusterDmaReadBusy" "" off \
    "ClusterDmaWriteBusy" "" off \
    "ClusterSampleCluster" "" off 2>&1 1>&3)

    return_value=$?

    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    choose_kl4_pu

    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    if [ -z "$pus" ]; then
        pus=$(seq -s ' ' 0 3)
    fi

    cluster_pus=$(dialog --clear --title "Mars Cluster Event Configuration" \
    --ok-label "Next" \
    --cancel-label "Back" \
    --checklist "Select pu-level signals to track:" 15 50 2 \
    "ClusterSamplePu" "" off \
    "ClusterSamplePC" "" off 2>&1 1>&3)

    return_value=$?

    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    choose_kl4_core

    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    if [ -z "$cores" ]; then
        cores=$(seq -s ' ' 0 15)
    fi

    cluster_cores=$(dialog --clear --title "Mars Cluster Event Configuration" \
    --ok-label "Add" \
    --cancel-label "Back" \
    --checklist "Select pu-level signals to track:" 15 50 1 \
    "ClusterSampleCore" "" off 2>&1 1>&3)

    return_value=$?

    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    insert_kl4_cluster_event
}

add_cluster_event() {
    choose_device
    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    if [ -z "$devices" ]; then
        show_add_error
        return
    fi

    choose_arch
    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    case $arch in
        3) add_kl3_cluster ;;
        4) add_kl4_cluster ;;
        *) show_add_error
            return ;;
    esac
}

##################################sdnn event##################################

insert_kl3_sdnn_event() {
    flag=0
    for cop in $sdnn_cops; do
        eval "events=\$$cop"
        if [ -n "$events" ]; then
            for device in $devices; do
                for unit in $units; do
                    for event in $events; do
                        bid=$(echo "\"board-id\": \"$device\",")
                        uid=$(echo "\"unit-id\": \"$unit\",")
                        name=$(echo "\"name\": \"$event\"")
                        echo "      {" >> "$file"
                        echo "          $bid" >> "$file"
                        echo "          $uid" >> "$file"
                        echo "          $name" >> "$file"
                        echo "      }," >> "$file"
                    done
                done
            done
            flag=1
        fi
    done

    if [ $flag == 1 ]; then
        show_add_result
    else
        show_add_error
    fi
}

add_kl3_sdnn() {
    choose_kl3_unit

    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    if [ -z "$units" ]; then
        show_add_error
        return
    fi

    sdnn_cops=$(dialog --clear --title "KL3 Sdnn Event Configuration" \
    --colors \
    --ok-label "Next" \
    --cancel-label "Back" \
    --checklist "Select Co-processor event to track(\Z1ALL cop enabled if select NONE\Zn):" 15 50 7 \
    "Ds" "" off \
    "Mac" "" off \
    "Rs" "" off \
    "Dmai" "" off \
    "Dmao" "" off \
    "Ew" "" off \
    "Sch" "" off 2>&1 1>&3)

    return_value=$?

    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    if [ -z "$sdnn_cops" ]; then
        sdnn_cops="Ds Mac Rs Dmai Dmao Ew Sch"
    fi

    for cop in $sdnn_cops; do
        if [[ "$cop" == "Sch" ]]; then
            output=$(dialog --clear --title "KL3 Sdnn Event Configuration" \
            --ok-label "Next" \
            --cancel-label "Back" \
            --checklist "Select ${cop} signals to track:" 15 50 1 \
            "Sdnn${cop}Duration" "" off 2>&1 1>&3)
        else
            output=$(dialog --clear --title "KL3 Sdnn Event Configuration" \
            --ok-label "Next" \
            --cancel-label "Back" \
            --checklist "Select ${cop} signals to track:" 15 50 3 \
            "Sdnn${cop}Duration" "" off \
            "Sdnn${cop}Xflag" "" off \
            "Sdnn${cop}Xtrace" "" off 2>&1 1>&3)
        fi
        return_value=$?
        if [ $return_value != $DIALOG_OK ]; then
            return
        fi
        eval "$cop='$output'"
    done

    insert_kl3_sdnn_event
}

insert_kl4_sdnn_event() {
    flag=0

    for cop in $sdnn_cops; do
        eval "events=\$$cop"
        if [ -n "$events" ]; then
            for device in $devices; do
                for unit in $units; do
                    for event in $events; do
                        bid=$(echo "\"board-id\": \"$device\",")
                        uid=$(echo "\"unit-id\": \"$unit\",")
                        name=$(echo "\"name\": \"$event\"")
                        echo "      {" >> "$file"
                        echo "          $bid" >> "$file"
                        echo "          $uid" >> "$file"
                        echo "          $name" >> "$file"
                        echo "      }," >> "$file"
                    done
                done
            done
            flag=1
        fi
    done

    if [ -n "$sdnn_coms" ]; then
        for device in $devices; do
            for unit in $units; do
                for event in $sdnn_coms; do
                    bid=$(echo "\"board-id\": \"$device\",")
                    uid=$(echo "\"unit-id\": \"$unit\",")
                    name=$(echo "\"name\": \"$event\"")
                    echo "      {" >> "$file"
                    echo "          $bid" >> "$file"
                    echo "          $uid" >> "$file"
                    echo "          $name" >> "$file"
                    echo "      }," >> "$file"
                done
            done
        done
        flag=1
    fi

    if [ $flag == 1 ]; then
        for device in $devices; do
            for unit in $units; do
                bid=$(echo "\"board-id\": \"$device\",")
                uid=$(echo "\"unit-id\": \"$unit\",")
                cid=$(echo "\"mode-id\": \"$sdnn_mode\",")
                name=$(echo "\"name\": \"SdnnTraceMode\"")
                echo "      {" >> "$file"
                echo "          $bid" >> "$file"
                echo "          $uid" >> "$file"
                echo "          $cid" >> "$file"
                echo "          $name" >> "$file"
                echo "      }," >> "$file"
            done
        done
        show_add_result
    else
        show_add_error
    fi
}

choose_kl4_sdnn_unit() {
    units=$(dialog --clear --title "Sdnn Event Configuration" \
    --colors \
    --ok-label "Next" \
    --cancel-label "Cancel" \
    --checklist "Choose unit Id(\Z1ALL units enabled if select NONE\Zn):" 15 50 6 \
    0 "unit 0" off \
    1 "unit 1" off \
    2 "unit 2" off \
    3 "unit 3" off \
    4 "unit 4" off \
    5 "unit 5" off 2>&1 1>&3)

    return_value=$?
}

add_kl4_sdnn() {
    choose_kl4_sdnn_unit

    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    if [ -z "$units" ]; then
        units=$(seq -s ' ' 0 1)
    fi

    sdnn_mode=$(dialog --clear --title "Sdnn Event Configuration" \
    --ok-label "Select" \
    --cancel-label "Cancel" \
    --menu "Choose Sdnn Trace Mode:" 15 50 4 \
    1 "Duration, Xflag, DMA Workinfo" \
    2 "Duration, Xflag, Xtrace, Sch Busy" \
    3 "Duration, Xflag, Work, Sch Busy" \
    4 "Duration, Sch Busy, Sch Xsignal, Sch Xwait"  2>&1 1>&3)

    return_value=$?

    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    if [ -z "$sdnn_mode" ]; then
        sdnn_mode="3"
    fi

    sdnn_cops=$(dialog --clear --title "Mars Sdnn Event Configuration" \
    --colors \
    --ok-label "Next" \
    --cancel-label "Back" \
    --checklist "Select Co-processor event to track(\Z1ALL cop enabled if select NONE\Zn):" 15 50 7 \
    "Ds" "" off \
    "Mac" "" off \
    "Rs" "" off \
    "Dmai" "" off \
    "Dmao" "" off \
    "Ew" "" off \
    "Ewlite" "" off 2>&1 1>&3)

    return_value=$?

    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    if [ -z "$sdnn_cops" ]; then
        sdnn_cops="Ds Mac Rs Dmai Dmao Ew Ewlite"
    fi

    for cop in $sdnn_cops; do
        checklist_options=(
            "Sdnn${cop}Duration" "" off
        )

        if [ "$sdnn_mode" -ne "4" ]; then
            checklist_options+=("Sdnn${cop}Xflag" "" off)
        fi

        if [ "$sdnn_mode" -eq "2" ]; then
            checklist_options+=("Sdnn${cop}Xtrace" "" off)
        elif [ "$sdnn_mode" -eq "3" ]; then
            checklist_options+=("Sdnn${cop}Work" "" off)
        fi

        output=$(dialog --clear --title "Mars Sdnn Event Configuration" \
            --ok-label "Next" \
            --cancel-label "Back" \
            --checklist "Select sdnn ${cop} signals to track:" 15 50 3 \
            "${checklist_options[@]}" 2>&1 1>&3)
        return_value=$?
        if [ $return_value != $DIALOG_OK ]; then
            return
        fi
        eval "$cop='$output'"
    done

    checklist_options=(
        "SdnnSampling" "" off
    )
    if [ "$sdnn_mode" -eq "1" ]; then
        checklist_options+=("SdnnDmaiWorkInfo" "" off)
        checklist_options+=("SdnnDmaoWorkInfo" "" off)
    else
        checklist_options+=("SdnnSchDuration" "" off)
    fi

    if [ "$sdnn_mode" -eq "4" ]; then
        checklist_options+=("SdnnSchXsignal" "" off)
        checklist_options+=("SdnnSchXwait" "" off)
    fi

    sdnn_coms=$(dialog --clear --title "Mars Sdnn Event Configuration" \
            --ok-label "Add" \
            --cancel-label "Back" \
            --checklist "Select remaining sdnn signals to track:" 15 50 4 \
            "${checklist_options[@]}" 2>&1 1>&3)
    return_value=$?
    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    insert_kl4_sdnn_event
}

add_sdnn_event() {
    choose_device
    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    if [ -z "$devices" ]; then
        show_add_error
        return
    fi

    choose_arch
    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    case $arch in
        3) add_kl3_sdnn ;;
        4) add_kl4_sdnn ;;
        *) show_add_error
            return ;;
    esac
}

##################################sdnn cluster event##################################

choose_kl4_sdnn_core() {
    cores=$(dialog --clear --title "Cluster Event Configuration" \
    --colors \
    --ok-label "Next" \
    --cancel-label "Cancel" \
    --checklist "Choose Core-level Id (\Z1ALL cores enabled if select NONE\Zn):" 15 50 8 \
    0 "core 0" off \
    1 "core 1" off \
    2 "core 2" off \
    3 "core 3" off \
    4 "core 4" off \
    5 "core 5" off \
    6 "core 6" off \
    7 "core 7" off 2>&1 1>&3)

    return_value=$?
}

insert_kl4_sdnn_cluster_event() {
    flag=0
    if [ -n "$cluster_events" ]; then
        for device in $devices; do
            for unit in $units; do
                for event in $cluster_events; do
                    bid=$(echo "\"board-id\": \"$device\",")
                    uid=$(echo "\"unit-id\": \"$unit\",")
                    name=$(echo "\"name\": \"$event\"")
                    echo "      {" >> "$file"
                    echo "          $bid" >> "$file"
                    echo "          $uid" >> "$file"
                    echo "          $name" >> "$file"
                    echo "      }," >> "$file"
                done
            done
        done
        flag=1
    fi

    if [ -n "$cluster_cores" ]; then
        for device in $devices; do
            for unit in $units; do
                for core in $cores; do
                    for event in $cluster_cores; do
                        bid=$(echo "\"board-id\": \"$device\",")
                        uid=$(echo "\"unit-id\": \"$unit\",")
                        cid=$(echo "\"core-id\": \"$core\",")
                        name=$(echo "\"name\": \"$event\"")
                        echo "      {" >> "$file"
                        echo "          $bid" >> "$file"
                        echo "          $uid" >> "$file"
                        echo "          $cid" >> "$file"
                        echo "          $name" >> "$file"
                        echo "      }," >> "$file"
                    done
                done
            done
        done
        flag=1
    fi

    if [ $flag == 1 ]; then
        show_add_result
    else
        show_add_error
    fi
}

add_kl4_sdnn_cluster() {
    choose_kl4_sdnn_unit

    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    if [ -z "$units" ]; then
        units=$(seq -s ' ' 0 1)
    fi

    cluster_events=$(dialog --clear --title "Mars Sdnn Cluster Event Configuration" \
    --ok-label "Next" \
    --cancel-label "Back" \
    --checklist "Select Sdnn cluster-level signals to track:" 15 50 5 \
    "SdnnClusterActive" "" off \
    "SdnnClusterGroup0LmWrite" "" off \
    "SdnnClusterGroup0LmRead" "" off \
    "SdnnClusterGroup1LmWrite" "" off \
    "SdnnClusterGroup1LmRead" "" off 2>&1 1>&3)

    return_value=$?

    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    choose_kl4_sdnn_core

    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    if [ -z "$cores" ]; then
        cores=$(seq -s ' ' 0 7)
    fi

    cluster_cores=$(dialog --clear --title "Mars Sdnn Cluster Event Configuration" \
    --ok-label "Add" \
    --cancel-label "Back" \
    --checklist "Select pu-level signals to track:" 15 50 8 \
    "SdnnClusterDuration" "" off \
    "SdnnClusterInstrFire" "" off \
    "SdnnClusterMfanceStall" "" off \
    "SdnnClusterSyncStall" "" off \
    "SdnnClusterBranchOrJrStall" "" off \
    "SdnnClusterSdnnWorking" "" off \
    "SdnnClusterXfanceDone" "" off \
    "SdnnClusterSdnnHandshake" "" off 2>&1 1>&3)

    return_value=$?

    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    insert_kl4_sdnn_cluster_event
}

add_sdnn_cluster_event() {
    choose_device
    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    if [ -z "$devices" ]; then
        show_add_error
        return
    fi

    choose_arch
    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    case $arch in
        4) add_kl4_sdnn_cluster ;;
        *) show_add_error
            return ;;
    esac
}

##################################cache event##################################

insert_kl4_cache_mode_event() {
    for device in $devices; do
        for unit in $units; do
            bid=$(echo "\"board-id\": \"$device\",")
            uid=$(echo "\"unit-id\": \"$unit\",")
            cid=$(echo "\"mode-id\": \"$cache_mode\",")
            name=$(echo "\"name\": \"CacheModeSelect\"")
            echo "      {" >> "$file"
            echo "          $bid" >> "$file"
            echo "          $uid" >> "$file"
            echo "          $cid" >> "$file"
            echo "          $name" >> "$file"
            echo "      }," >> "$file"
        done
    done
}

insert_cache_event() {
    if [ -n "$cache_event" ]; then
        for device in $devices; do
            for unit in $units; do
                for event in $cache_event; do
                    bid=$(echo "\"board-id\": \"$device\",")
                    cid=$(echo "\"unit-id\": \"$unit\",")
                    name=$(echo "\"name\": \"$event\"")
                    echo "      {" >> "$file"
                    echo "          $bid" >> "$file"
                    echo "          $cid" >> "$file"
                    echo "          $name" >> "$file"
                    echo "      }," >> "$file"
                done
            done
        done
        show_add_result
    else
        show_add_error
    fi

    if [[($arch == 4) && ($cache_mode != 0)]]; then
        insert_kl4_cache_mode_event
    fi
}

add_kl3_cache() {
    units=(0)

    cache_event=$(dialog --clear --title "KL3 HBM Event Configuration" \
    --ok-label "Next" \
    --cancel-label "Back" \
    --checklist "Select HBM signals to track:" 15 50 2 \
    "CacheWrThroughputHbm" "" off \
    "CacheRdThroughputHbm" "" off 2>&1 1>&3)

    return_value=$?
}

choose_kl4_cache_unit() {
    units=$(dialog --clear --title "Mars Cache/HBM Event Configuration" \
    --colors \
    --ok-label "Next" \
    --cancel-label "Cancel" \
    --checklist "Choose cache wrapper unit Id(\Z1ALL units enabled if select NONE\Zn):" 15 50 12 \
    0 "unit 0" off \
    1 "unit 1" off \
    2 "unit 2" off \
    3 "unit 3" off \
    4 "unit 4" off \
    5 "unit 5" off \
    6 "unit 6" off \
    7 "unit 7" off \
    8 "unit 8" off \
    9 "unit 9" off \
    10 "unit 10" off \
    11 "unit 11" off 2>&1 1>&3)

    return_value=$?
}

add_kl4_cache() {

    cache_mode=$(dialog --clear --title "Mars Cache/HBM Event Configuration" \
    --ok-label "Select" \
    --cancel-label "Cancel" \
    --menu "Choose Cache Trace Mode:" 15 50 5 \
    0 "Throughput" \
    1 "HBM AXI" \
    2 "L3 AXI" \
    3 "HBM & L3 AXI" \
    4 "Merge AXI"  2>&1 1>&3)

    return_value=$?

    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    if [ -z "$cache_mode" ]; then
        cache_mode="0"
    fi

    if [ "$cache_mode" -eq "0" ]; then
        units=(0)
    else
        choose_kl4_cache_unit
        if [ $return_value != $DIALOG_OK ]; then
            return
        fi
        if [ -z "$units" ]; then
            units=$(seq -s ' ' 0 12)
        fi
    fi

    if [ "$cache_mode" -eq "0" ]; then
        cache_event=$(dialog --clear --title "Mars Cache/HBM Event Configuration" \
        --ok-label "Next" \
        --cancel-label "Back" \
        --checklist "Select Cache/HBM signals to track:" 15 50 4 \
        "CacheWrThroughputHbm" "" off \
        "CacheWrThroughputHbm" "" off \
        "CacheRdThroughputL3" "" off \
        "CacheRdThroughputL3" "" off 2>&1 1>&3)
    elif ["$cache_mode" -eq "4"]; then
        cache_event=$(dialog --clear --title "Mars Cache/HBM Event Configuration" \
        --ok-label "Next" \
        --cancel-label "Back" \
        --checklist "Select Cache/HBM signals to track:" 15 50 4 \
        "CacheWrRateHbmTotal" "" off \
        "CacheRdRateHbmTotal" "" off \
        "CacheWrRateL3Total" "" off \
        "CacheRdRateL3Total" "" off 2>&1 1>&3)
    else
        cache_event=$(dialog --clear --title "Mars Cache/HBM Event Configuration" \
        --ok-label "Next" \
        --cancel-label "Back" \
        --checklist "Select Cache/HBM signals to track:" 15 50 16 \
        "CacheWrRate" "" off \
        "CacheWrRdy0" "" off \
        "CacheWrPart" "" off \
        "CacheRdRate" "" off \
        "CacheRdRdy0" "" off \
        "CacheAwRate" "" off \
        "CacheAwRdy0" "" off \
        "CacheArRate" "" off \
        "CacheArRdy0" "" off \
        "CacheBRate" "" off \
        "CacheBRdy0" "" off \
        "CacheAwOstdAccuLo" "" off \
        "CacheAwOstdAccuHi" "" off \
        "CacheArOstdAccuLo" "" off \
        "CacheArOstdAccuHi" "" off \
        "CacheCycle" "" off 2>&1 1>&3)
    fi

    return_value=$?
}

add_cache_event() {
    choose_device
    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    if [ -z "$devices" ]; then
        show_add_error
        return
    fi

    choose_arch
    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    case $arch in
        3) add_kl3_cache ;;
        4) add_kl4_cache ;;
        *) show_add_error
            return ;;
    esac

    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    insert_cache_event
}

##################################sdma event##################################

insert_kl4_sdma_channel_event() {
    for device in $devices; do
        bid=$(echo "\"board-id\": \"$device\",")
        tmp=$(expr $sdma_ch - 1)
        cid=$(echo "\"channel-id\": \"$tmp\",")
        name=$(echo "\"name\": \"SdmaChannelSelect\"")
        echo "      {" >> "$file"
        echo "          $bid" >> "$file"
        echo "          $cid" >> "$file"
        echo "          $name" >> "$file"
        echo "      }," >> "$file"
    done
}

insert_sdma_event() {
    if [ -n "$sdma_event" ]; then
        for device in $devices; do
            for event in $sdma_event; do
                bid=$(echo "\"board-id\": \"$device\",")
                name=$(echo "\"name\": \"$event\"")
                echo "      {" >> "$file"
                echo "          $bid" >> "$file"
                echo "          $name" >> "$file"
                echo "      }," >> "$file"
            done
        done
        show_add_result
    else
        show_add_error
    fi

    if [[($arch == 4) && ($sdma_ch != 0)]]; then
        insert_kl4_sdma_channel_event
    fi
}

add_kl4_sdma() {

    sdma_ch=$(dialog --clear --title "Mars SDMA Event Configuration" \
    --colors \
    --ok-label "Select" \
    --cancel-label "Cancel" \
    --menu "Choose SDMA Trace Channel(\Z1Select one or count for ALL by default\Zn):" 15 50 19 \
    0 "all units" \
    1 "unit 0" \
    2 "unit 1" \
    3 "unit 2" \
    4 "unit 3" \
    5 "unit 4" \
    6 "unit 5" \
    7 "unit 6" \
    8 "unit 7" \
    9 "unit 8" \
    10 "unit 9" \
    11 "unit 10" \
    12 "unit 11" \
    13 "unit 12" \
    14 "unit 13" \
    15 "unit 14" \
    16 "unit 15" \
    17 "unit 16" \
    18 "unit 17" 2>&1 1>&3)

    return_value=$?

    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    if [ -z "$sdma_ch" ]; then
        sdma_ch="0"
    fi

    sdma_event=$(dialog --clear --title "Mars SDMA Event Configuration" \
    --ok-label "Next" \
    --cancel-label "Back" \
    --checklist "Select SDMA signals to track:" 15 50 11 \
    "SdmaAr0Issue" "" off \
    "SdmaAw0Issue" "" off \
    "SdmaB0Issue" "" off \
    "SdmaR0Issue" "" off \
    "SdmaW0Issue" "" off \
    "SdmaAr1Issue" "" off \
    "SdmaAw1Issue" "" off \
    "SdmaB1Issue" "" off \
    "SdmaR1Issue" "" off \
    "SdmaW1Issue" "" off \
    "SdmaSampling" "" off 2>&1 1>&3)

    return_value=$?
}

add_sdma_event() {
    choose_device
    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    if [ -z "$devices" ]; then
        show_add_error
        return
    fi

    choose_arch
    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    case $arch in
        4) add_kl4_sdma ;;
        *) show_add_error
            return ;;
    esac

    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    insert_sdma_event
}

##################################pcie event##################################

insert_pcie_event() {
    if [ -n "$pcie_event" ]; then
        for device in $devices; do
            for unit in $units; do
                for event in $pcie_event; do
                    bid=$(echo "\"board-id\": \"$device\",")
                    uid=$(echo "\"unit-id\": \"$unit\",")
                    name=$(echo "\"name\": \"$event\"")
                    echo "      {" >> "$file"
                    echo "          $bid" >> "$file"
                    echo "          $uid" >> "$file"
                    echo "          $name" >> "$file"
                    echo "      }," >> "$file"
                done
            done
        done
        show_add_result
    else
        show_add_error
    fi
}

choose_kl4_pcie_unit() {
    units=$(dialog --clear --title "Mars PCIe/C2C Event Configuration" \
    --colors \
    --ok-label "Next" \
    --cancel-label "Cancel" \
    --checklist "Choose PCIe/C2C units(\Z1ALL units enabled if select NONE\Zn):" 15 50 5 \
    0 "PCIe Subsystem" off \
    1 "C2C 0" off \
    2 "C2C 1" off \
    3 "C2C 2" off \
    4 "C2C 3" off  2>&1 1>&3)

    return_value=$?
}

add_kl4_pcie() {

    choose_kl4_pcie_unit
    if [ $return_value != $DIALOG_OK ]; then
        return
    fi
    if [ -z "$units" ]; then
        units=$(seq -s ' ' 0 4)
    fi

    pcie_event=$(dialog --clear --title "Mars PCIe/C2C Event Configuration" \
    --ok-label "Next" \
    --cancel-label "Back" \
    --checklist "Select PCIe/C2C signals to track:" 15 50 11 \
    "PcieAr0Issue" "" off \
    "PcieAw0Issue" "" off \
    "PcieB0Issue" "" off \
    "PcieR0Issue" "" off \
    "PcieW0Issue" "" off \
    "PcieAr1Issue" "" off \
    "PcieAw1Issue" "" off \
    "PcieB1Issue" "" off \
    "PcieR1Issue" "" off \
    "PcieW1Issue" "" off \
    "PcieSampling" "" off 2>&1 1>&3)

    return_value=$?
}

add_pcie_event() {
    choose_device
    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    if [ -z "$devices" ]; then
        show_add_error
        return
    fi

    choose_arch
    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    case $arch in
        4) add_kl4_pcie ;;
        *) show_add_error
            return ;;
    esac

    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    insert_pcie_event
}

##################################smmu event##################################

insert_smmu_event() {
    flag=0

    for event in $tcu_events $tcu_samps $tbu_events $tbu_samps; do
        for device in $devices; do
            for unit in $units; do
                bid=$(echo "\"board-id\": \"$device\",")
                uid=$(echo "\"unit-id\": \"$unit\",")
                name=$(echo "\"name\": \"$event\"")
                echo "      {" >> "$file"
                echo "          $bid" >> "$file"
                echo "          $uid" >> "$file"
                echo "          $name" >> "$file"
                echo "      }," >> "$file"
            done
        done
        flag=1
    done

    if [ $flag == 1 ]; then
        show_add_result
    else
        show_add_error
    fi
}

choose_kl4_smmu_unit() {
    units=$(dialog --clear --title "Mars SMMU Event Configuration" \
    --colors \
    --ok-label "Next" \
    --cancel-label "Cancel" \
    --checklist "Choose SMMU TCU units(\Z1ALL units enabled if select NONE\Zn):" 15 50 8 \
    0 "TCU 0" off \
    1 "TCU 1" off \
    2 "TCU 2" off \
    3 "TCU 3" off \
    4 "TCU 4" off \
    5 "TCU 5" off \
    6 "TCU 6" off \
    7 "TCU 7" off  2>&1 1>&3)

    return_value=$?
}

add_kl4_smmu() {

    choose_kl4_smmu_unit

    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    if [ -z "$units" ]; then
        units=$(seq -s ' ' 0 7)
    fi

    tcu_events=$(dialog --clear --title "Mars SMMU Event Configuration" \
    --ok-label "Next" \
    --cancel-label "Back" \
    --checklist "Select TCU signals to track:" 15 50 13 \
    "SmmuTcuReadValid" "" off \
    "SmmuTcuReadReturn" "" off \
    "SmmuTcuTidFull" "" off \
    "SmmuTcuTidEmpty" "" off \
    "SmmuTcuCdCacheFull" "" off \
    "SmmuTcuCdCacheEmpty" "" off \
    "SmmuTcuCdCacheConflict" "" off \
    "SmmuTcuHptCacheFull" "" off \
    "SmmuTcuHptCacheEmpty" "" off \
    "SmmuTcuHptCacheConflict" "" off \
    "SmmuTcuMptCacheFull" "" off \
    "SmmuTcuMptCacheEmpty" "" off \
    "SmmuTcuMptCacheConflict" "" off 2>&1 1>&3)

    return_value=$?

    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    tcu_samps=$(dialog --clear --title "Mars SMMU Event Configuration" \
    --ok-label "Next" \
    --cancel-label "Back" \
    --checklist "Select TCU counters to track:" 15 50 4 \
    "SmmuTcuCdCacheCounter" "" off \
    "SmmuTcuHptCacheCounter" "" off \
    "SmmuTcuMptCacheCounter" "" off \
    "SmmuTcuAxiReadCounter" "" off  2>&1 1>&3)

    return_value=$?

    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    tbu_events=$(dialog --clear --title "Mars SMMU Event Configuration" \
    --ok-label "Next" \
    --cancel-label "Back" \
    --checklist "Select TCU signals to track:" 15 50 9 \
    "SmmuTbuRTransactionFull" "" off \
    "SmmuTbuRTransactionEmpty" "" off \
    "SmmuTbuWTransactionFull" "" off \
    "SmmuTbuWTransactionEmpty" "" off \
    "SmmuTbuWBufferFull" "" off \
    "SmmuTbuWBufferEmpty" "" off \
    "SmmuTlbMsmrFull" "" off \
    "SmmuTlbMsmrEmpty" "" off \
    "SmmuTlbSetConflict" "" off  2>&1 1>&3)

    return_value=$?

    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    tbu_samps=$(dialog --clear --title "Mars SMMU Event Configuration" \
    --ok-label "Next" \
    --cancel-label "Back" \
    --checklist "Select TBU counters to track:" 15 50 4 \
    "SmmuTbuTlbReqCounter" "" off \
    "SmmuTbuTlbHitCounter" "" off \
    "SmmuTbuTlbMissCounter" "" off \
    "SmmuTbuTlbPtwCounter" "" off  2>&1 1>&3)

    return_value=$?

    if [ $return_value != $DIALOG_OK ]; then
        return
    fi
}

add_smmu_event() {
    choose_device
    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    if [ -z "$devices" ]; then
        show_add_error
        return
    fi

    choose_arch
    if [ $return_value != $DIALOG_OK ]; then
        return
    fi

    case $arch in
        4) add_kl4_smmu ;;
        *) show_add_error
            return ;;
    esac

    insert_smmu_event
}

##################################main##################################

file="xpu-prof.gen-settings.json"
echo "{" > "$file"
echo "  \"signal\":[" >> "$file"

while true; do
    exec 3>&1
    show_menu
    case $return_value in
        $DIALOG_CANCEL)
            break ;;
    esac
    case $menu_choice in
        1) add_driver_event ;;
        2) add_sse_event ;;
        3) add_cluster_event ;;
        4) add_sdnn_event ;;
        5) add_sdnn_cluster_event ;;
        6) add_cache_event ;;
        7) add_sdma_event ;;
        8) add_pcie_event ;;
        9) add_smmu_event ;;
        10) break ;;
    esac
    exec 3>&-
done

sed -i '$ d' "$file" # 删除最后一行
echo "      }" >> "$file"
echo "  ]" >> "$file"
echo "}" >> "$file"

clear